<?php
/*
 * Plugin Name: Folder List
 * Description: Listet alle Dateien in einem angegebenen Verzeichnis innerhalb des WordPress-Verzeichnisses, und zeigt sie auf einer Seite per Shortcode an. Zusätzlich werden Downloads gezählt und im Backend angezeigt.
 * Version: 2.5.1
 * Author: codebase.software by Achim Schmidt
 * Author URI: https://codebase.software
 * License: Commercial
 * License URI: https://codebase.software/license
 * Text Domain: folder-list
 * Domain Path: /languages
 * Requires at least: 6.0
 * Requires PHP: 7.4
 * 
 * © 2025 Codebase.software by Achim Schmidt
 */

 
 if ( ! defined( 'ABSPATH' ) ) exit;
 
 /* -------------------------------------------------------------
	Table name pattern: [prefix]_cbs_fli_folder_list_downloads
 ------------------------------------------------------------- */
 function folder_list_get_table_name() {
	 global $wpdb;
	 return $wpdb->prefix . 'cbs_fli_folder_list_downloads';
 }
 
 /* -------------------------------------------------------------
	Create database table if not exists
 ------------------------------------------------------------- */
 function folder_list_activate() {
	 global $wpdb;
 
	 $table   = folder_list_get_table_name();
	 $collate = $wpdb->get_charset_collate();
 
	 $sql = "CREATE TABLE $table (
		 id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
		 file_path text NOT NULL,
		 file_name varchar(255) NOT NULL,
		 folder_path text NOT NULL,
		 download_count bigint(20) unsigned NOT NULL DEFAULT 0,
		 first_download datetime DEFAULT NULL,
		 last_download datetime DEFAULT NULL,
		 PRIMARY KEY (id),
		 KEY file_name (file_name(191))
	 ) $collate;";
 
	 require_once ABSPATH . 'wp-admin/includes/upgrade.php';
	 dbDelta($sql);
 }
 register_activation_hook(__FILE__, 'folder_list_activate');
 
 /* -------------------------------------------------------------
	Ensure columns exist for older installs
 ------------------------------------------------------------- */
 function folder_list_maybe_upgrade_table() {
	 global $wpdb;
	 $table = folder_list_get_table_name();
 
	 $exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table));
	 if ($exists !== $table) return;
 
	 $cols = $wpdb->get_col("DESC $table", 0);
	 if (!$cols) return;
 
	 if (!in_array('first_download', $cols, true)) {
		 $wpdb->query("ALTER TABLE $table ADD first_download datetime DEFAULT NULL AFTER download_count");
	 }
 
	 if (!in_array('last_download', $cols, true)) {
		 $wpdb->query("ALTER TABLE $table ADD last_download datetime DEFAULT NULL AFTER first_download");
	 }
 
	 // initialize missing first_download
	 $wpdb->query("UPDATE $table SET first_download = last_download WHERE first_download IS NULL AND last_download IS NOT NULL");
 }
 add_action('plugins_loaded', 'folder_list_maybe_upgrade_table');
 
 /* -------------------------------------------------------------
	Shortcode: List files from a folder
	[directory_file_list server_path="/abs/path"] or [directory_file_list relative_path="wp-content/uploads/files"]
 ------------------------------------------------------------- */
 function directory_file_list_shortcode($atts) {
 
	 $atts = shortcode_atts([
		 'server_path'   => '',
		 'relative_path' => '',
	 ], $atts, 'directory_file_list');
 
	 if (!$atts['server_path'] && !$atts['relative_path']) {
		 return 'Please specify a path.';
	 }
 
	 $server_path = $atts['server_path']
		 ? rtrim($atts['server_path'], '/\\') . DIRECTORY_SEPARATOR
		 : rtrim(ABSPATH . ltrim($atts['relative_path'], '/\\'), '/\\') . DIRECTORY_SEPARATOR;
 
	 if (!is_dir($server_path)) {
		 return 'Invalid path.';
	 }
 
	 $files = scandir($server_path);
	 if (!$files) {
		 return 'No files found.';
	 }
 
	 $list = array_filter($files, function($f) use ($server_path) {
		 return !in_array($f, ['.', '..'], true) && is_file($server_path . $f);
	 });
	 if (!$list) {
		 return 'No files found.';
	 }
 
	 $out  = '<ul class="directory-file-list">';
	 $root = realpath(ABSPATH);
 
	 foreach ($list as $file_name) {
		 $abs  = $server_path . $file_name;
		 $real = realpath($abs);
 
		 if (!$real || !$root || strpos($real, $root) !== 0) {
			 continue;
		 }
 
		 $rel     = ltrim(str_replace($root, '', $real), '/\\');
		 $encoded = rawurlencode(base64_encode($rel));
 
		 $url = add_query_arg([
			 'folder_list_download' => 1,
			 'file'                 => $encoded,
		 ], home_url('/'));
 
		 $out .= '<li><a href="' . esc_url($url) . '">' . esc_html($file_name) . '</a></li>';
	 }
	 $out .= '</ul>';
 
	 // Footer: 25px Abstand, 2px Linie, Schriftgröße 11px
	 $out .= '<div class="folder-list-footer" style="margin-top:25px;padding-top:8px;border-top:2px solid #ccc;font-size:16px;text-align:center;color:#666666;opacity:0.85;">
		 Made with love by
		 <a href="https://codebase.software" target="_blank" rel="noopener noreferrer" style="color:inherit;font-weight:600;text-decoration:none;">
			 codebase.software
		 </a>
	 </div>';
 
	 return $out;
 }
 add_shortcode('directory_file_list', 'directory_file_list_shortcode');
 
 /* -------------------------------------------------------------
	Download handler (logging + sending file)
 ------------------------------------------------------------- */
 function folder_list_maybe_handle_download() {
	 if (!isset($_GET['folder_list_download'], $_GET['file'])) {
		 return;
	 }
	 if (is_admin() && !wp_doing_ajax()) {
		 return;
	 }
 
	 $decoded = base64_decode(rawurldecode((string) $_GET['file']), true);
	 if (!$decoded) {
		 wp_die('Invalid parameter.');
	 }
 
	 $relative = ltrim(str_replace(['../', '..\\'], '', $decoded), '/\\');
	 $root     = realpath(ABSPATH);
	 $file     = $root ? realpath($root . DIRECTORY_SEPARATOR . $relative) : false;
 
	 if (!$root || !$file || strpos($file, $root) !== 0 || !is_file($file)) {
		 wp_die('File not found.');
	 }
 
	 folder_list_register_download($relative, $file);
	 folder_list_send_file($file);
	 exit;
 }
 add_action('init', 'folder_list_maybe_handle_download');
 
 /* -------------------------------------------------------------
	Register download
 ------------------------------------------------------------- */
 function folder_list_register_download($relative, $absolute) {
	 global $wpdb;
 
	 $table  = folder_list_get_table_name();
	 $file   = basename($absolute);
	 $folder = dirname($relative);
	 if ($folder === '.') {
		 $folder = '';
	 }
 
	 $now = current_time('mysql');
 
	 $id = $wpdb->get_var(
		 $wpdb->prepare("SELECT id FROM $table WHERE file_path = %s", $relative)
	 );
 
	 if ($id) {
		 $wpdb->query(
			 $wpdb->prepare(
				 "UPDATE $table SET download_count = download_count + 1, last_download = %s WHERE id = %d",
				 $now,
				 $id
			 )
		 );
	 } else {
		 $wpdb->insert(
			 $table,
			 [
				 'file_path'      => $relative,
				 'file_name'      => $file,
				 'folder_path'    => $folder,
				 'download_count' => 1,
				 'first_download' => $now,
				 'last_download'  => $now,
			 ],
			 ['%s','%s','%s','%d','%s','%s']
		 );
	 }
 }
 
 /* -------------------------------------------------------------
	Send file to browser
 ------------------------------------------------------------- */
 function folder_list_send_file($file) {
	 if (!file_exists($file)) {
		 wp_die('File missing.');
	 }
 
	 if (ob_get_length()) {
		 @ob_end_clean();
	 }
 
	 nocache_headers();
	 $mime = wp_check_filetype(basename($file));
	 $type = $mime['type'] ?: 'application/octet-stream';
 
	 header('Content-Type: ' . $type);
	 header('Content-Disposition: attachment; filename="' . basename($file) . '"');
	 header('Content-Length: ' . filesize($file));
 
	 readfile($file);
	 exit;
 }
 
 /* -------------------------------------------------------------
	Admin menu
 ------------------------------------------------------------- */
 function folder_list_register_admin_menu() {
	 add_menu_page(
		 'Folder List Downloads',
		 'Folder List',
		 'manage_options',
		 'folder-list-downloads',
		 'folder_list_render_admin_page',
		 'dashicons-download',
		 80
	 );
 }
 add_action('admin_menu', 'folder_list_register_admin_menu');
 
 /* -------------------------------------------------------------
	Admin CSS
 ------------------------------------------------------------- */
 function folder_list_admin_styles() {
	 echo '<style>
		 .folder-list-folder-header {
			 margin-top: 2em;
			 margin-bottom: 0.4em;
			 display: flex;
			 gap: 1em;
			 align-items: baseline;
		 }
		 .folder-list-folder-heading,
		 .folder-list-folder-total {
			 font-size: 1.25rem;
			 font-weight: 600;
		 }
		 .folder-list-separator {
			 font-size: 1.25rem;
			 font-weight: 600;
		 }
	 </style>';
 }
 add_action('admin_head-toplevel_page_folder-list-downloads', 'folder_list_admin_styles');
 
 /* -------------------------------------------------------------
	Admin statistics page
 ------------------------------------------------------------- */
 function folder_list_render_admin_page() {
	 if (!current_user_can('manage_options')) {
		 wp_die('No permission.');
	 }
 
	 global $wpdb;
	 $table     = folder_list_get_table_name();
	 $downloads = $wpdb->get_results("SELECT * FROM $table ORDER BY folder_path ASC, file_name ASC");
 
	 echo '<div class="wrap"><h1>Folder List – Download Statistics</h1>';
 
	 if (!$downloads) {
		 echo '<p>No downloads recorded yet.</p></div>';
		 return;
	 }
 
	 $folder_totals = [];
	 foreach ($downloads as $row) {
		 $folder = (string) $row->folder_path;
		 $folder_totals[$folder] = ($folder_totals[$folder] ?? 0) + (int) $row->download_count;
	 }
 
	 $current = null;
	 $open    = false;
 
	 foreach ($downloads as $row) {
 
		 $folder    = $row->folder_path;
		 $file      = esc_html($row->file_name);
		 $path      = esc_html($row->file_path);
		 $count     = (int) $row->download_count;
		 $first_raw = $row->first_download;
		 $last_raw  = $row->last_download;
 
		 $first = $first_raw ?: '–';
		 $last  = $last_raw  ?: '–';
 
		 $days = '–';
		 if ($first_raw && $last_raw) {
			 $d1 = strtotime($first_raw);
			 $d2 = strtotime($last_raw);
			 if ($d2 >= $d1) {
				 $days = floor(($d2 - $d1) / 86400) + 1;
			 }
		 }
 
		 if ($folder !== $current) {
 
			 if ($open) {
				 echo '</tbody></table>';
			 }
 
			 $current = $folder;
			 $label   = $folder !== '' ? esc_html($folder) : 'Root';
			 $total   = $folder_totals[$folder] ?? 0;
 
			 echo '<div class="folder-list-folder-header">
					 <div class="folder-list-folder-heading">' . $label . '</div>
					 <div class="folder-list-separator">•</div>
					 <div class="folder-list-folder-total">Total: ' . $total . '</div>
				   </div>';
 
			 echo '<table class="widefat fixed striped"><thead><tr>
					 <th>Filename</th>
					 <th>Downloads</th>
					 <th>First Download</th>
					 <th>Last Download</th>
					 <th>Days</th>
					 <th>Path (relative)</th>
				   </tr></thead><tbody>';
 
			 $open = true;
		 }
 
		 echo '<tr>
				 <td>' . $file . '</td>
				 <td>' . $count . '</td>
				 <td>' . $first . '</td>
				 <td>' . $last . '</td>
				 <td>' . $days . '</td>
				 <td><code>' . $path . '</code></td>
			   </tr>';
	 }
 
	 if ($open) {
		 echo '</tbody></table>';
	 }
 
	 echo '</div>';
 }
