#!/bin/zsh
#
# Script Name: backup_multi_folders.sh
# Description: Fasst mehrere Ordner in EINEM ZIP-Archiv zusammen (mit Zeitstempel-Präfix).
# Author: codebase.software by Achim Schmidt
# License: GPLv3 (https://www.gnu.org/licenses/gpl-3.0.txt)
# If you like it, buy me a coffee:  https://buymeacoffee.com/achimschmidt
# Requires: zip
#
# Nutzungs-Hinweis:
# 1) Skript ausführbar machen: chmod +x scripte/backup_multi_folders.sh
# 2) Start: ./scripte/backup_multi_folders.sh
# 
#
# ───────────────────────────── Benutzer­anpassungen ────────────────────────────
# Zielordner für das Archiv (wird bei Bedarf erstellt)
DESTINATION="/Volumes/Name der Festplatte/1. Ordner/2. Ordner/3. Ordner/ziel-Ordner/"

# Zeitstempel automatisch erzeugen (Format: JJJJ-MM-TT_hh-mm)
TIMESTAMP=$(date +"%Y-%m-%d_%H%M")

# Dein händisch gesetzter Name (ohne .zip)
BASE_NAME="gewünschter Dateiname"

# Fertiger Archivname: Zeitstempel + Name (muss .zip sein)
ARCHIVE_NAME="${TIMESTAMP}_${BASE_NAME}.zip"

# Liste der zu sichernden Ordner (beliebig erweiterbar)
SOURCES=(
  "/Volumes/Name der Festplatte/1. Ordner/2. Ordner/3. Ordner/Quell-Ordner/"
  "/Volumes/Name der Festplatte/1. Ordner/2. Ordner/3. Ordner/Quell-Ordner/"
  "/Volumes/Name der Festplatte/1. Ordner/2. Ordner/3. Ordner/Quell-Ordner/"
)


# true = vorhandenes Archiv überschreiben; false = abbrechen, wenn vorhanden
OVERWRITE=false
# ───────────────────────────────────────────────────────────────────────────────

set -euo pipefail

# ───────────────────────────── Hilfsfunktionen ────────────────────────────────
err()  { print -P "❌ %F{red}$1%f" >&2; }
ok()   { print -P "✅ %F{green}$1%f"; }
info() { print -P "ℹ️  %F{blue}$1%f"; }

# ───────────────────────────── Vorprüfungen ───────────────────────────────────
# zip vorhanden?
if ! command -v zip >/dev/null 2>&1; then
  err "zip ist nicht installiert. Bitte nachinstallieren (z. B. 'brew install zip' oder via Paketmanager)."
  exit 1
fi

# Archivname prüfen
if [[ -z "${ARCHIVE_NAME:-}" ]]; then
  err "ARCHIVE_NAME ist leer. Prüfe BASE_NAME und TIMESTAMP."
  exit 1
fi
if [[ "${ARCHIVE_NAME:l}" != *.zip ]]; then
  err "ARCHIVE_NAME muss auf .zip enden. Aktuell: ${ARCHIVE_NAME}"
  exit 1
fi

# Quellliste prüfen
if [[ "${#SOURCES[@]}" -eq 0 ]]; then
  err "Die Liste SOURCES ist leer. Bitte mindestens einen Ordner eintragen."
  exit 1
fi

# Existieren alle Quellordner?
for SRC in "${SOURCES[@]}"; do
  if [[ ! -d "$SRC" ]]; then
    err "Quellordner nicht gefunden: $SRC"
    exit 1
  fi
done

# Zielordner anlegen + Schreibrecht prüfen
mkdir -p "$DESTINATION"
if [[ ! -d "$DESTINATION" ]]; then
  err "Zielordner konnte nicht erstellt werden: $DESTINATION"
  exit 1
fi
if [[ ! -w "$DESTINATION" ]]; then
  err "Kein Schreibrecht im Zielordner: $DESTINATION"
  exit 1
fi

ARCHIVE_PATH="${DESTINATION%/}/${ARCHIVE_NAME}"

# Existierendes Archiv behandeln
if [[ -f "$ARCHIVE_PATH" ]]; then
  if [[ "$OVERWRITE" == true ]]; then
    info "Bestehendes Archiv wird überschrieben: $ARCHIVE_PATH"
    rm -f "$ARCHIVE_PATH"
  else
    err "Archiv existiert bereits: $ARCHIVE_PATH  (OVERWRITE=true setzen oder anderen Namen wählen)"
    exit 1
  fi
fi

# ───────────────────────────── Archiv erstellen ───────────────────────────────
info "Erstelle ZIP-Archiv: $ARCHIVE_PATH"

# Jeden Ordner so hinzufügen, dass im Archiv nur der Ordnername als Wurzel erscheint
for SRC in "${SOURCES[@]}"; do
  PARENT_DIR="${SRC%/*}"
  ENTRY_NAME="${SRC##*/}"
  info "Füge hinzu: ${SRC}  →  /${ENTRY_NAME}/"
  (
    cd "$PARENT_DIR" || { err "Kann nicht in $PARENT_DIR wechseln."; exit 1; }
    # -r: rekursiv, -q: ruhig (quiet), -y: symbolische Links als Links speichern (Default-Verhalten beibehalten)
    zip -rq "$ARCHIVE_PATH" "$ENTRY_NAME"
  )
done

ok "Backup erfolgreich erstellt: $ARCHIVE_PATH"
