#!/bin/zsh
#
# Script Name: backup_multi_folders_interactive.sh
# Description: Interaktives ZIP-Backup: Quellen vor dem Packen aus Liste entfernen + Archivname anpassen.
## Author: codebase.software by Achim Schmidt
# License: GPLv3 (https://www.gnu.org/licenses/gpl-3.0.txt)
# If you like it, buy me a coffee:  https://buymeacoffee.com/achimschmidt
# Requires: zip
#
# Datei ausführbar machen: chmod +x  scripte/backup_multi_folders_interactive.sh
## Datei starten: ./scripte/backup_multi_folders_interactive.sh
#
# ───────────────────────────── Benutzer­anpassungen (Defaults) ─────────────────
# Zielordner für das Archiv (wird bei Bedarf erstellt)
DESTINATION="/Volumes/Name der Festplatte/1. Ordner/2. Ordner/3. Ordner/Ziel-Ordner/"

# Zeitstempel automatisch erzeugen (Format: JJJJ-MM-TT_hh-mm)
TIMESTAMP=$(date +"%Y-%m-%d_%H-%M")

# Dein händisch gesetzter Basisname (ohne .zip) – kann interaktiv geändert werden
BASE_NAME="mein-archiv"

# Liste der zu sichernden Ordner (beliebig erweiterbar) – kann interaktiv eingegrenzt werden
SOURCES=(
  "/Volumes/Name der Festplatte/1. Ordner/2. Ordner/3. Ordner/Quell-Ordner/"
  "/Volumes/Name der Festplatte/1. Ordner/2. Ordner/3. Ordner/Quell-Ordner/"
  "/Volumes/Name der Festplatte/1. Ordner/2. Ordner/3. Ordner/Quell-Ordner/"
)

# true = vorhandenes Archiv überschreiben; false = abbrechen, wenn vorhanden
OVERWRITE=false
# ───────────────────────────────────────────────────────────────────────────────

set -euo pipefail

# ───────────────────────────── Hilfsfunktionen ────────────────────────────────
err()  { print -P "❌ %F{red}$1%f" >&2; }
ok()   { print -P "✅ %F{green}$1%f"; }
info() { print -P "ℹ️  %F{blue}$1%f"; }
warn() { print -P "⚠️  %F{yellow}$1%f"; }

has_command() { command -v "$1" >/dev/null 2>&1; }

print_sources() {
  local i=1
  print -P "%F{cyan}Aktuelle Quellverzeichnisse:%f"
  for s in "${SOURCES[@]}"; do
    printf "  %2d) %s\n" "$i" "$s"
    ((i++))
  done
}

filter_sources_interactively() {
  while true; do
    echo
    print_sources
    echo
    print -P "%F{cyan}Eingrenzen:%f Geben Sie die Nummern der zu ENTFERNENDEN Einträge an (z. B. '2 5 7' oder '2,5,7')."
    print -P "Drücken Sie einfach %F{green}Enter%f, um alle zu behalten. Mit %F{yellow}a%f leeren Sie die Liste. Mit %F{yellow}q%f brechen Sie ab."
    echo -n "> "
    local input
    IFS= read -r input || true

    case "${input:l}" in
      "")  # nichts löschen
        break
        ;;
      q)   # abbrechen
        err "Abbruch durch Nutzer."
        exit 130
        ;;
      a)   # alle löschen
        SOURCES=()
        break
        ;;
      *)
        input="${input//,/ }"
        local to_remove=(${=input})
        local valid=true
        local n=${#SOURCES[@]}
        for idx in "${to_remove[@]}"; do
          if [[ ! "$idx" =~ '^[0-9]+$' ]]; then
            valid=false; warn "Ungültiger Eintrag: '$idx' (nur Nummern erlaubt)"
          elif (( idx < 1 || idx > n )); then
            valid=false; warn "Nummer außerhalb des Bereichs: $idx (1..$n)"
          fi
        done
        if [[ "$valid" == false ]]; then
          continue
        fi
        local new_sources=()
        local i=1
        for s in "${SOURCES[@]}"; do
          local drop=false
          for rem in "${to_remove[@]}"; do
            if (( i == rem )); then drop=true; break; fi
          done
          if [[ "$drop" == false ]]; then
            new_sources+=("$s")
          fi
          ((i++))
        done
        SOURCES=("${new_sources[@]}")
        ;;
    esac

    echo
    print -P "%F{cyan}Neue Auswahl:%f"
    if (( ${#SOURCES[@]} == 0 )); then
      warn "Es befinden sich nun KEINE Quellen mehr in der Liste."
      break
    fi
    print_sources
    echo -n "Übernehmen? (j/n) > "
    local yn
    IFS= read -r yn || true
    if [[ "${yn:l}" == "j" || -z "${yn}" ]]; then
      break
    fi
  done
}

adjust_archive_name_interactively() {
  local default_name="${TIMESTAMP}_${BASE_NAME}.zip"
  print -P "\n%F{cyan}Aktueller Archivname:%f ${default_name}"
  print -P "Neuen %F{green}Basisnamen%f eingeben (ohne .zip) oder Enter drücken, um zu übernehmen:"
  echo -n "> "
  local new_base
  IFS= read -r new_base || true
  if [[ -n "${new_base:-}" ]]; then
    BASE_NAME="$new_base"
  fi
  ARCHIVE_NAME="${TIMESTAMP}_${BASE_NAME}.zip"
}

confirm_or_exit() {
  echo
  print_sources
  print -P "\n%F{cyan}Zielordner:%f $DESTINATION"
  print -P "%F{cyan}Archivname:%f  $ARCHIVE_NAME"
  echo -n "Fortfahren? (j/n) > "
  local yn
  IFS= read -r yn || true
  if [[ "${yn:l}" != "j" && -n "${yn}" ]]; then
    err "Abbruch durch Nutzer."
    exit 130
  fi
}

# ───────────────────────────── Vorprüfungen ───────────────────────────────────
if ! has_command zip; then
  err "zip ist nicht installiert. Bitte installieren (z. B. 'brew install zip')."
  exit 1
fi

if (( ${#SOURCES[@]} == 0 )); then
  err "Die Liste SOURCES ist leer. Bitte mindestens einen Ordner eintragen."
  exit 1
fi

for SRC in "${SOURCES[@]}"; do
  if [[ ! -d "$SRC" ]]; then
    err "Quellordner nicht gefunden: $SRC"
    exit 1
  fi
done

mkdir -p "$DESTINATION" || { err "Zielordner konnte nicht erstellt werden: $DESTINATION"; exit 1; }
[[ -w "$DESTINATION" ]] || { err "Kein Schreibrecht im Zielordner: $DESTINATION"; exit 1; }

# ───────────────────────────── Interaktive Schritte ───────────────────────────
filter_sources_interactively
if (( ${#SOURCES[@]} == 0 )); then
  err "Keine Quellen ausgewählt. Vorgang abgebrochen."
  exit 1
fi

adjust_archive_name_interactively
confirm_or_exit

ARCHIVE_PATH="${DESTINATION%/}/${ARCHIVE_NAME}"

if [[ -f "$ARCHIVE_PATH" ]]; then
  if [[ "$OVERWRITE" == true ]]; then
    info "Bestehendes Archiv wird überschrieben: $ARCHIVE_PATH"
    rm -f "$ARCHIVE_PATH"
  else
    err "Archiv existiert bereits: $ARCHIVE_PATH  (OVERWRITE=true setzen oder anderen Namen wählen)"
    exit 1
  fi
fi

# ───────────────────────────── Archiv erstellen ───────────────────────────────
info "Erstelle ZIP-Archiv: $ARCHIVE_PATH"
for SRC in "${SOURCES[@]}"; do
  PARENT_DIR="${SRC%/*}"
  ENTRY_NAME="${SRC##*/}"
  info "Füge hinzu: ${SRC}  →  /${ENTRY_NAME}/"
  (
    cd "$PARENT_DIR" || { err "Kann nicht in $PARENT_DIR wechseln."; exit 1; }
    zip -rq "$ARCHIVE_PATH" "$ENTRY_NAME"
  )
done

ok "Backup erfolgreich erstellt: $ARCHIVE_PATH"